/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/


/* ---------------------------------------------------------------
 * File: xanalyze.c
 *       general Analyzer functions 
 * -----------------------------------------------------------------*/

#include <xtypedef.h>

#include <xiocommo.h>
#include <xpciapi.h>
#include <xregcons.h>
#include <xregxdir.h>
#include <xregx10.h>
#include <xregx11.h>
#include <xsession.h>
#include <xversion.h>
#include <xaddrmap.h>

#if 0				/* HL, unnecessary includes??? */
#include <xpci.h>
#include <timeout.h>
#endif

/* Pointer to analyzer database */
#define DB_ANALYZER (bx_handlearray[handle].db->Analyzer)

/* Value of generic property */
#define GENERIC(prop) (DB_ANALYZER.GenProp[prop])

/********************************************************************
  PCI Analyzer Generic Property Programming Functions ***************
********************************************************************/

/*---------------------------------------------------------------------------*
 * BestXAnalyzerGenGet()
 *
 * Purpose: gets a generic property from DB 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXAnalyzerGenGet(
  bx_handletype handle,
  bx_agentype prop,
  bx_int32 * val
)
{
  BX_DECLARE_FUNCNAME("BestXAnalyzerGenGet [agenget]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_ANALYZER_GEN, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
     
    /* get property */
    *val=(bx_int32)GENERIC(prop);
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXAnalyzerGenSet()
 *
 * Purpose: sets a generic property in host-DB 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXAnalyzerGenSet(
  bx_handletype handle,
  bx_agentype prop,
  bx_int32 val)

{
  BX_DECLARE_FUNCNAME("BestXAnalyzerGenSet [agenset]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_ANALYZER_GEN, (bx_int32)prop, val));
     
    /* set property */
    GENERIC(prop)=(bx_int32)val;
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXAnalyzerGenDefaultSet()
 *
 * Purpose: Defaults all analyzer generic properties in host-DB
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXAnalyzerGenDefaultSet(
  bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXAnalyzerGenDefaultSet [agendefset]");

  BX_TRY_VARS_NO_PROG;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_ANALYZER_GEN, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_ANALYZER_GEN,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXAnalyzerGenSet(handle,
              ParamInfo->proptyp.agenprop,
              ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXAnalyzerGenProg()
 *
 * Purpose: Programs properties from host DB to card
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXAnalyzerGenProg(
  bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXAnalyzerGenProg [agenprog]");

  BX_TRY_VARS_NO_PROG;
  bx_int32 val1;

  BX_TRY_BEGIN
  {
    /* BX_AGEN_SEM10 */
    BX_TRY(BestXAnalyzerGenGet(handle,BX_AGEN_SEM10,&val1));

    /* 12 bit HW register;
     A value of zero in HW (HW default:0xfff) means 0x1000
     A value of one would trigger the rule immediately */
     
     if (BestXHasMephisto(handle))
     {
       BX_TRY(BestXDirectRegWrite(handle,BX_REG_SEM_10_PL_REG_M,sizeof(bx_int16),val1));
     }
     else
     {
       BX_TRY(BestXDirectRegWrite(handle,BX_REG_SEM_10_PL_REG_F,sizeof(bx_int16),val1));
     }

    /* BX_AGEN_SEM11 */
    BX_TRY(BestXAnalyzerGenGet(handle,BX_AGEN_SEM11,&val1));
    /* 20 bit HW register;
     A value of zero in HW (HW default: undefined) means 0x100000
     A value of one would trigger the rule immediately */
    
     if (BestXHasMephisto(handle))
     {
       BX_TRY(BestXDirectRegWrite(handle,BX_REG_SEM_11_PL_REG_M,sizeof(bx_int32),val1));
     }
     else
     {
       BX_TRY(BestXDirectRegWrite(handle,BX_REG_SEM_11_PL_REG_F,sizeof(bx_int32),val1));
     }
  }

  BX_ERRETURN(BX_TRY_RET);
}
/*---------------------------------------------------------------------------*
 * BestXAnalyzerGenRead()
 *
 * Purpose: Reads properties from card to host DB 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXAnalyzerGenRead(
  bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXAnalyzerGenRead [agenread]");

  BX_TRY_VARS_NO_PROG;
  bx_int32 val1;

  BX_TRY_BEGIN
  {
    /* BX_AGEN_SEM10 */
    if (BestXHasMephisto(handle))
    {
      BX_TRY(BestXDirectRegRead(handle,BX_REG_SEM_10_PL_REG_M,sizeof(bx_int16),&val1));
    }
    else
    {
      BX_TRY(BestXDirectRegRead(handle,BX_REG_SEM_10_PL_REG_F,sizeof(bx_int16),&val1));
    }
    /* 12 bit HW register;
     A value of zero in HW (HW default:0xfff) means 0x1000
     A value of one would trigger the rule immediately */
    
    BX_TRY(BestXAnalyzerGenSet(handle,BX_AGEN_SEM10,val1));

    /* BX_AGEN_SEM11 */
    if (BestXHasMephisto(handle))
    {
      BX_TRY(BestXDirectRegRead(handle,BX_REG_SEM_11_PL_REG_M,sizeof(bx_int32),&val1));
    }
    else
    {
      BX_TRY(BestXDirectRegRead(handle,BX_REG_SEM_11_PL_REG_F,sizeof(bx_int32),&val1));
    }
    
    /* 20 bit HW register;
     A value of zero in HW (HW default !) means 0x100000
     A value of one would trigger the rule immediately */
    
    BX_TRY(BestXAnalyzerGenSet(handle,BX_AGEN_SEM11,val1));
  }

  BX_ERRETURN(BX_TRY_RET);
}


/********************************************************************
  PCI Analyzer Clock switching
********************************************************************/

/*---------------------------------------------------------------------------*
 * BestXAnalyzerProgMode()
 *
 * Purpose: Switches analyzer into programming mode (DBI clock). 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXAnalyzerProgMode(bx_handletype handle,bx_int32 SoftForce)
{
  BX_DECLARE_FUNCNAME("BestXAnalyzerProgMode [aprogmode]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {

      /* Layout of sync-mechanism:
         1. MEPHISTO_OFFSET+0x30: 
            -32 bit control register, RW
            -only bit 0 implemented: 0=Progmode,1=Runmode
         2. MEPHISTO_OFFSET+0x34: 
            -32 bit ready-,sync- and force register:
            -Bit 0 is sync bit from control register
            -Bit 1 is ready bit: 0=ready, 1=busy
            -Writing an arbitrary value forces bit 0 from 
             0x20 to 0x24
      */

      BX_TRY(BestXSyncRegWrite(handle,
                             
                             BX_REG_TR_CLK_SWITCH_STAT_REG,
                             sizeof(bx_int32),
                             0x2,  /* Mask for Bit 1    */
                             0x0,  /* 0 in Bit 1=Ready  */
                             
                             BX_REG_TR_CLK_SWITCH_REG,
                             sizeof(bx_int32),
                             0x1, /* Mask for bit 0 */
                             0,   /* 0=ProgMode     */
                             
                             BX_REG_TR_CLK_SWITCH_STAT_REG,
    
                             BX_REG_TR_CLK_SWITCH_STAT_REG, /* Force register */
                             sizeof(bx_int32),
                             SoftForce
            ));
                              
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXAnalyzerRunMode()
 *
 * Purpose: Switches analyzer into run mode (DBI clock). 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXAnalyzerRunMode(bx_handletype handle,bx_int32 SoftForce)
{
  BX_DECLARE_FUNCNAME("BestXAnalyzerRunMode [arunmode]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
      /* Layout of sync-mechanism:
         1. MEPHISTO_OFFSET+0x30: 
            -32 bit control register, RW
            -only bit 0 implemented: 0=Progmode,1=Runmode
         2. MEPHISTO_OFFSET+0x34: 
            -32 bit ready-,sync- and force register:
            -Bit 0 is sync bit from control register
            -Bit 1 is ready bit: 0=ready, 1=busy
            -Writing an arbitrary value forces bit 0 from 
             0x20 to 0x24
      */

      /* Switch to runmode */
      BX_TRY(BestXSyncRegWrite(handle,
                             
                             BX_REG_TR_CLK_SWITCH_STAT_REG,
                             sizeof(bx_int32),
                             0x2,  /* Mask for Bit 1    */
                             0x0,  /* 0 in Bit 1=Ready  */
                             
                             BX_REG_TR_CLK_SWITCH_REG,
                             sizeof(bx_int32),
                             0x1, /* Mask for bit 0 */
                             1,   /* 1=RunMode      */
                             
                             BX_REG_TR_CLK_SWITCH_STAT_REG,
    
                             BX_REG_TR_CLK_SWITCH_STAT_REG, /* Force register */
                             sizeof(bx_int32),
                             SoftForce
                             ));
                              
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXAnalyzerIsInProgMode()
 *
 * Purpose: Checks, wether CAPI thinks that card is in prog-mode or not 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXAnalyzerIsInProgMode(bx_handletype handle,
                                             bx_int32 *ProgMode)
{
  BX_DECLARE_FUNCNAME("BestXAnalyzerIsInProgMode [aisinprogmode]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    if (ProgMode)
    {
      BX_TRY(BestXDirectRegRead(handle,BX_REG_TR_CLK_SWITCH_STAT_REG,sizeof(bx_int32),ProgMode));
      *ProgMode=((*ProgMode)&1?0:1);
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXAnalyzerRun()
 *
 * Purpose: Starts analyzer trace                           
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXAnalyzerRun(bx_handletype handle)
{
  return BestXTraceRun(handle);
}


/*---------------------------------------------------------------------------*
 * BestXAnalyzerStop()
 *
 * Purpose: Stops analyzer trace                           
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXAnalyzerStop(bx_handletype handle)
{
  return BestXTraceStop(handle);
}
